/*

Copyright 2019 NXP
All rights reserved.
SPDX-License-Identifier: BSD-3-Clause

*/

#include "EmbeddedTypes.h"
#include "arm_math.h" /* CMSIS ARM lib */
#define FLOAT_32_T_DEFINED /* Prevent float_32_t type refedefinition in xcvr */
#define FLOAT_64_T_DEFINED /* Prevent float_64_t type refedefinition in xcvr */
#include "fsl_device_registers.h"
#include "fsl_os_abstraction.h"
#include "fsl_port.h"
#include "fsl_gpio.h"
#include "genfsk_interface.h"
#include "lcl_utils.h"
#include "nxp_xcvr_ext_ctrl.h"
#include "lcl_aoa.h"
#include "dma_capture.h"
#include "mathfp.h"
#include "genfsk_utils.h"

/*! *********************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
********************************************************************************** */
#ifndef WEAK
#if defined(__GNUC__)
#define WEAK __attribute__ ((__weak__))
#elif defined(__IAR_SYSTEMS_ICC__)
#define WEAK __weak
#endif
#endif

#define LCL_AOA_IRQ           PORTB_PORTC_IRQn
#define LCL_AOA_LANT_SW_PIN   (31U)  // lant_sw is connected to dummy PTC31
#define LCL_AOA_LANT_SW_PORT  (PORTC)
#define LCL_AOA_LANT_SW_GPIO  (GPIOC)

#define LCL_OTA_TO_PHY_DELAY_US    (4U) /* HW delay  */
#define LCL_OTA_TO_PHY_DELAY_B0_US (6U) /* HW delay  */
#define LCL_LANT_TO_GPIO_DELAY_US  (2U) /* SW Delay measured btw ANT GPIOS toggling and lant_sw trigger on falling edge = 1.876us */
#define LCL_DMA_TO_LCL_DELAY_US    (LCL_OTA_TO_PHY_DELAY_US + LCL_LANT_TO_GPIO_DELAY_US) // in us
#define LCL_DMA_TO_LCL_DELAY_B0_US (LCL_OTA_TO_PHY_DELAY_B0_US + LCL_LANT_TO_GPIO_DELAY_US) // in us
#define LCL_DMA_DELAY_INCREMENT_US (6U) /* 6 us contingency is added to compensate the measured delay btw wideband IQ path (where AA match occurs) and narrowband IQ path (where capture occurs) */
#define LCL_IQ_SAMPLE_LENGTH       (4U) /* 32 bits */
#define LCL_SPINT                  (2U) /* LCL SPINT interval in us */

#define Q14_SCALING_FACTOR    (16384)
#define Q15_SCALING_FACTOR    (32768)

#ifdef AOA_ALGO_INTEGRATION
#include "lcl_aoa_algo.h"

#define QR_EPSILON (0.01)
#define QR_NUM_ITERATION (16)
#define AVERAGING_WINDOW (16) // hard-coded for now; should be 2/3 * Nseg?

#define Q6_SCALING_FACTOR     (64)
#define Q11_SCALING_FACTOR    (2048)
#define Q12_SCALING_FACTOR    (4096)

#define LCL_AOA_NSRC (2)

float Iin[LCL_AOA_NUM_MAX_ANTENNAS*32];
float Qin[LCL_AOA_NUM_MAX_ANTENNAS*32];
#endif

#define LCL_AOA_INVALID_ANGLE (0x8000)
#define LCL_AOA_INVALID_AQI   (0x8000)

#define LCL_AOA_NUM_CAPTURE_BEFORE_AGC_LOCK  (2U)
#define LCL_AOA_NUM_CAPTURE_BEFORE_AGC_UNLOCK (2U)
#define LCL_AOA_MAX_CAPTURE_DURING_AGC_LOCK (6U)
#define LCL_AOA_AGC_RECORD_IDX(agc_idx, n) \
  do { \
    (agc_idx) = LCL_GET_AGC_INDEX; \
    agcLockingTable[(n)] = (uint8_t)(agc_idx); \
  } while (FALSE);
#define LCL_AOA_AGC_COMPUTE_MEAN(mean, num) \
  do { \
    (mean) = 0; \
    LCL_ASSERT((num) !=0); \
    for (uint32_t i=0; i<(num); i++) \
    { \
        (mean) = (mean) + agcLockingTable[i]; \
    } \
    (mean) = (mean) / (num); \
  } while (FALSE);


  
/************************************************************************************
*************************************************************************************
* Private types declarations
*************************************************************************************
************************************************************************************/
typedef enum
{
    LCL_AOA_STATE_IDLE = 0,
    LCL_AOA_STATE_CONFIGURED,
    LCL_AOA_STATE_CAPTURING,
    LCL_AOA_STATE_ERROR
}lclAoAState_t;

/*! @brief Calibration for a given Antenna
           - phase is provided in input 
           - cos and sin are computed once when config is received 
*/ 
typedef struct 
{
    Q15_t phase; /* Phase calibration in radians [0; 2pi) scaled to [0; +1) in Q15 format */
    Q15_t cos; /* cos(phase) in Q15 format */
    Q15_t sin; /* cos(phase) in Q15 format */
} lclAoaAntCalib_t;

typedef struct 
{
     /* Internal variables */
    uint32_t             antennaIndex; /*!< index in pattern. */
    uint32_t             currentPolarization; /*!< current polarization */
    bool_t               calibrationDone;
    
    /* Config */
    uint8_t             sampleRate; /*!< Sample rate in Msamples/s. 4 for KW37/38. */
    uint8_t             nMpSrc; /*!< Number of sources for MP algo (1 to 3) */
    uint16_t            dmaTriggerDelay; /*!< Delay in us between trigger (AA found) and start of IQ capture. */
    lclAoaCaptureTriggerType_t captureTrigger; /*!< Trigger used to start DMA capture */
    lclAntennaConfig_t  ant_cnf; /*!< Antenna configuration. */
    lclAoACallBack_t    captureCallback; /*!< Callback called when capture is completed. */
    lclAoAAntSwitchCallBack_t antSwitchCallback; /*!< Callback called when antenna switch is triggered. */
    lclAoaAgcMode_t     agcMode;            /*!< AGC mode */
    uint8_t             forcedAgcIndex;     /*!< AGC index if forced mode configured. Range is [0-26] */
    bool_t              disableCFOest;      /*!< Disable CFO estimation */
    uint8_t             byteStreamPatternSize; /*!< Size of byte stream pattern to be matched. Only relevant when captureTrigger is set to LCL_AOA_CAPTURE_TRIGGER_PATTERN_MATCH */
    uint8_t             byteStreamPattern[LCL_AOA_MAX_PATTERN_SIZE]; /*!< bit stream pattern to be matched. Only relevant when captureTrigger is set LCL_AOA_CAPTURE_TRIGGER_PATTERN_MATCH */
    lclAoaAntCalib_t    antCalibration[LCL_AOA_NUM_MAX_ANTENNAS]; /*!< Phase calibration per antenna */
    
    /* State */
    lclAoAState_t       state;
    lclAoaAgcState_t    agcState; /*!< AGC state */
    
    /* Buffers */
    uint16_t            numIqBuffTemp;
    uint32_t            *iqBuffTemp_p;
    int16_t             *iqBuff_p;
    lclAoaCaptureInd_t  *captureInd_p;

} lclAoaInternal_t;

/*! *********************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
********************************************************************************** */
static lclAoaInternal_t lclAoaGlob;

#ifdef ENABLE_PROFILING
volatile unsigned long dwt_preproc;
volatile unsigned long dwt_MP;
#endif

static uint8_t agcLockingTable[LCL_AOA_MAX_CAPTURE_DURING_AGC_LOCK];

/************************************************************************************
*************************************************************************************
* Private prototypes
*************************************************************************************
************************************************************************************/
LCL_STATIC void LclAoaAGCupdate(lclAoaAgcEvent_t event);


/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/
static void LclAoADeinitAntennaSwitching(void)
{
    (void)DisableIRQ(LCL_AOA_IRQ);
}

static void LclAoAIrqHandler(void)
{
    /* Increase antenna index and change polarization if needed */
    if (++lclAoaGlob.antennaIndex == lclAoaGlob.ant_cnf.numAnt)
    {
        lclAoaGlob.antennaIndex = 0;
        if (lclAoaGlob.captureInd_p->polarConfig == gAntPolDynamic)
        {
            if (++lclAoaGlob.currentPolarization >= LCL_AOA_MAX_POLARIZATIONS) 
            {
                lclAoaGlob.currentPolarization = 0;
            }
        }
    }
    
    /* Call antSwitchCallback in order for the application to actually control antenna change */
    /* using whatever mean (GPIOs, TPMs, etc.) */
    if (lclAoaGlob.antSwitchCallback != NULL)
    {
        lclAoaGlob.antSwitchCallback(lclAoaGlob.ant_cnf.antPattern[lclAoaGlob.antennaIndex], (lclAntennaPolarization_t)lclAoaGlob.currentPolarization);
    }
    
    GPIO_PortClearInterruptFlags(LCL_AOA_LANT_SW_GPIO, ((uint32_t)1U) << LCL_AOA_LANT_SW_PIN);
}

static void LclAoAInitAntennaSwitching(void)
{
    lclAntennaSwitchingPatternConfig_t lcl_switch;
    xcvrLclStatus_t status;
    uint16_t delay;
    uint16_t aoaRadioVersion = XCVR_ReadRadioVer();
    
    gpio_pin_config_t antPinConfig =
    {
        kGPIO_DigitalInput, 0,
    };
    
    /* Dummy PORTC31 is used for lant_sw, see RM */
    PORT_SetPinMux(LCL_AOA_LANT_SW_PORT, LCL_AOA_LANT_SW_PIN, kPORT_MuxAsGpio); 
    
    /* Configure interrupt on falling edge */
    PORT_SetPinInterruptConfig(LCL_AOA_LANT_SW_PORT, LCL_AOA_LANT_SW_PIN, kPORT_InterruptFallingEdge);
    
    GPIO_PinInit(LCL_AOA_LANT_SW_GPIO, LCL_AOA_LANT_SW_PIN, &antPinConfig);
    
    /* Enable IRQ and install handler */
    OSA_InstallIntHandler((uint32_t)LCL_AOA_IRQ, LclAoAIrqHandler);

    (void)EnableIRQ(LCL_AOA_IRQ);

    lclAoaGlob.antennaIndex = 0;
    
    /* Compute antenna switching start delay */
    delay = lclAoaGlob.dmaTriggerDelay + lclAoaGlob.ant_cnf.guardPeriodDuration;
    
    if (((uint16_t)XCVR_RADIO_GEN_3P5_B0) == aoaRadioVersion)
    {
        /* On B0, add an extra switchingPeriod to make sure capture on first switching pattern happens on main antenna  */
        delay = delay + lclAoaGlob.ant_cnf.switchingPeriodDuration - LCL_DMA_TO_LCL_DELAY_B0_US;
    }
    else
    {
        delay -= LCL_DMA_TO_LCL_DELAY_US;
    }
    
    /* Configure antenna switching parameters */
    lcl_switch.numberOfSwitchesInPattern = lclAoaGlob.ant_cnf.numOfpatternRepetition * lclAoaGlob.ant_cnf.numAnt;
    lcl_switch.lantSwInvert = false;
    lcl_switch.lantSwWiggle = false;
    lcl_switch.txTrig = lclTxTriggerSoftware; /* unused */
    
    switch (lclAoaGlob.captureTrigger)
    {
    case LCL_AOA_CAPTURE_TRIGGER_PATTERN_MATCH:
        lcl_switch.rxTrig = lclRxTriggerPatternFound;
        break;
    case LCL_AOA_CAPTURE_TRIGGER_CTE_PRESENT:
        lcl_switch.rxTrig = lclRxTriggerCtePresent;
        LCL_ASSERT(FALSE); /* Not supported yet */
        break;
    case LCL_AOA_CAPTURE_TRIGGER_AA_MATCH:
    default:
        lcl_switch.rxTrig = lclRxTriggerAccessAddressFound;
        break;
    }

    lcl_switch.samplesPerInterval = (LCL_SPINT * lclAoaGlob.sampleRate); // LCL_SPINT (=2) us
    
    lcl_switch.lowPeriodDuration = (uint8_t)(lclAoaGlob.ant_cnf.switchingPeriodDuration / LCL_SPINT);
    lcl_switch.highPeriodDuration = (uint8_t)(lclAoaGlob.ant_cnf.samplingPeriodDuration / LCL_SPINT);
    lcl_switch.triggerDelay = (uint8_t)(delay / LCL_SPINT);
    lcl_switch.triggerOffset = ((uint8_t) (delay % LCL_SPINT)) * lclAoaGlob.sampleRate;

    status = XCVR_LCL_AntennaSwitchInit(gLclRxMode, &lcl_switch);
    LCL_ASSERT(status == gXcvrLclStatusSuccess);
    
    /* Configure the pattern to be matched if enabled */
    if (lclAoaGlob.captureTrigger == LCL_AOA_CAPTURE_TRIGGER_PATTERN_MATCH)
    {
        lclPatternMatchConfig_t lclPatternMatchConfig;

        switch (lclAoaGlob.byteStreamPatternSize)
        {
        case 4:
          lclPatternMatchConfig.numBytes = lclPatternMatchNumBytes4;
          break;
        case 5:
          lclPatternMatchConfig.numBytes = lclPatternMatchNumBytes5;
          break;
        case 6:
        case 7:
          lclPatternMatchConfig.numBytes = lclPatternMatchNumBytes6;
          break;
        case 8:
        default:
          lclPatternMatchConfig.numBytes = lclPatternMatchNumBytes8;
          break;
        }
        /* Swap pattern bytes bit per bit */
        GENFSK_SwapBytes(lclAoaGlob.byteStreamPattern, lclPatternMatchConfig.pattern, lclAoaGlob.byteStreamPatternSize);
        
        status =  XCVR_LCL_PatternMatchInit(gLclRxMode, &lclPatternMatchConfig);
        LCL_ASSERT(status == gXcvrLclStatusSuccess);
    }
    
    NOT_USED(status);
}

static void LclAoAStartAntennaSwitching(void)
{    
    xcvrLclStatus_t status;
    
    /* Enable RX mode antenna switching */
    status = XCVR_LCL_AntennaSwitchEn(false, true);
    LCL_ASSERT(status == gXcvrLclStatusSuccess);
    
    /* Enable pattern matching if needed */
    if (lclAoaGlob.captureTrigger == LCL_AOA_CAPTURE_TRIGGER_PATTERN_MATCH)
    {
        status = XCVR_LCL_PatternMatchEn(gLclRxMode);
        LCL_ASSERT(status == gXcvrLclStatusSuccess);
    }
    
    NOT_USED(status);
    
    if (lclAoaGlob.captureInd_p->polarConfig != gAntPolDynamic)
    {
        lclAoaGlob.currentPolarization = (uint32_t)lclAoaGlob.captureInd_p->polarConfig;
    }
    else
    {
        lclAoaGlob.currentPolarization = (uint32_t)gAntPol_90;
    }
    
    /* Enable default antenna */
    if (lclAoaGlob.antSwitchCallback != NULL)
    {
        lclAoaGlob.antSwitchCallback(lclAoaGlob.ant_cnf.mainAntenna, (lclAntennaPolarization_t)lclAoaGlob.currentPolarization);
    }
}

static void LclAoAStopAntennaSwitching(void)
{
    xcvrLclStatus_t status;
    
    /* Disable antenna switching */
    status = XCVR_LCL_AntennaSwitchEn(false, false);
    LCL_ASSERT(status == gXcvrLclStatusSuccess);
    
    status = XCVR_LCL_PatternMatchDisable(gLclRxMode);
    LCL_ASSERT(status == gXcvrLclStatusSuccess);
    NOT_USED(status);
    
    /* Re-Initialize antenna switching: reset index and select main */
    lclAoaGlob.antennaIndex = 0;
    
    /* and restore default polarization */
    if (lclAoaGlob.captureInd_p->polarConfig == gAntPolDynamic)
    {
        lclAoaGlob.currentPolarization = (uint32_t)gAntPol_90;
    }
    
    if (lclAoaGlob.antSwitchCallback != NULL)
    {
        lclAoaGlob.antSwitchCallback(lclAoaGlob.ant_cnf.mainAntenna, (lclAntennaPolarization_t)lclAoaGlob.currentPolarization);
    }
}

/************************************************/
//LCOV_EXCL_START
#if defined(LCL_COVERAGE)
WEAK void LclAoATweakIqSample(lclAoaCaptureInd_t *pInd) {}
#endif
//LCOV_EXCL_STOP

/************************************************/

/* Check  IQ samples with regards to saturation */
LCL_STATIC void LclAoAIqSampleCheck(lclAoaCaptureInd_t *pInd)
{
    uint16_t *pBuffIn;
    uint16_t patterdIdx, k;
    bool_t saturationFound = FALSE;
    uint32_t temp;

    LCL_ASSERT(pInd != NULL);

    /* Look for I / Q saturation and record it for further use */
    for (patterdIdx = 0u ; patterdIdx < pInd->antPatternLength; patterdIdx++)
    {
        pBuffIn = (uint16_t *)pInd->ant[patterdIdx].iqBuff_p;
        pInd->ant[patterdIdx].antMeasStatus = gAoAMeasSuccess;
        
        for (k = 0u ; k < pInd->ant[patterdIdx].numIq; k++)
        {
            /* Check max and min thresholds */
            if ((pBuffIn[2u*k] == LCL_MAX_IQ_VALUE) || (pBuffIn[2u*k] == LCL_MIN_IQ_VALUE) || 
                (pBuffIn[2u*k+1u] == LCL_MAX_IQ_VALUE) || (pBuffIn[2u*k+1u] == LCL_MIN_IQ_VALUE))
            {
                saturationFound = TRUE;
                pInd->ant[patterdIdx].antMeasStatus = gAoAMeasFailSaturation;
                break; // and examine next antenna
            }
        }  
    }
    
    /* Record misc. information  */
    pInd->agc_index = (uint8_t)LCL_GET_AGC_INDEX;
    pInd->agcState = lclAoaGlob.agcState;
    temp = LCL_GET_CFO_EST;
    pInd->cfo_est = (int16_t)temp;
    temp = LCL_GET_RSSI;
    pInd->rssi = (int8_t)temp;
    pInd->timestamp = (uint64_t)GENFSK->TIMESTAMP;
    
    if (saturationFound)
    {
        pInd->measStatus = gAoAMeasFailSaturation;
    }
}

/* Update AGC state machine if needed */
static void LclAoAUpdateAGCstateMachine(lclAoaCaptureInd_t *captureInd_p)
{
    uint16_t patterdIdx;
    bool_t saturationFound = FALSE;
    
    /* Look for saturation  */
    for (patterdIdx = 0u ; patterdIdx < captureInd_p->antPatternLength; patterdIdx++)
    {
        if (captureInd_p->ant[patterdIdx].antMeasStatus == gAoAMeasFailSaturation)
        {
            saturationFound = TRUE;
            break;
        }
    }
    
    /* Update AGC state machine with SATURATION_DETECTED or OK events */
    if (lclAoaGlob.agcMode == LCL_AOA_AGC_MODE_MANUAL)
    {
        if (saturationFound)
        {
            LclAoaAGCupdate(LCL_AOA_AGC_EVENT_SATURATION_DETECTED);
        }
        else
        {
            LclAoaAGCupdate(LCL_AOA_AGC_EVENT_CAPTURE_OK);
        }
    }
}

/* Estimate DC offset using reference period */
static void LclAoAEstimateDcOffset(lclAoaCaptureInd_t *captureInd_p)
{
    /* Estimate DC-offset */
    uint16_t numSamplesForDc; 
    uint32_t i;
    // it is safe to accumulate using int32 for reference periods less than 128 us (no overflow)
    int32_t meanI = 0;
    int32_t meanQ = 0; 
    
    if (lclAoaGlob.ant_cnf.guardPeriodDuration != 0U)
    {
        numSamplesForDc = lclAoaGlob.ant_cnf.guardPeriodDuration * lclAoaGlob.sampleRate;
    }
    else
    {
        /* in case guardPeriodDuration is set to zero...*/
        numSamplesForDc = lclAoaGlob.ant_cnf.samplingPeriodDuration * lclAoaGlob.sampleRate;
    }
    LCL_ASSERT(numSamplesForDc != 0U);

    for (i=0; i<numSamplesForDc; i++)
    {
        meanI += (int32_t)captureInd_p->rawIqBuff_p[2u*i];
        meanQ += (int32_t)captureInd_p->rawIqBuff_p[2u*i+1u];
    }
    meanI /= ((int32_t)numSamplesForDc);
    meanQ /= ((int32_t)numSamplesForDc);
    
    /* Populate output struct */
    captureInd_p->dcOffset.meanI = (int16_t)meanI;
    captureInd_p->dcOffset.meanQ = (int16_t)meanQ;
}

/* Remap IQ data and remove DC offset */
/* IQ samples are output to iqBuff_p[] buffer */
static void LclAoARemapIQdata(lclAoaCaptureInd_t *captureInd_p)
{
    uint32_t patternIdx;
    uint32_t j = 0;
    int16_t *buff_in_p;
    int16_t *buff_out_p;

    buff_out_p = lclAoaGlob.iqBuff_p;
    
    for (patternIdx = 0u ; patternIdx < captureInd_p->antPatternLength; patternIdx++)
    {    
        buff_in_p = captureInd_p->ant[patternIdx].iqBuff_p;
        
        captureInd_p->ant[patternIdx].iqBuff_p = buff_out_p; // swap iq buffer pointer
        
        for (j=0; j < captureInd_p->ant[patternIdx].numIq; j++)
        {
            /* Move IQs, remove guard period, remove DC offset and remove 4 unused LSBs */
            *buff_out_p++ = (*buff_in_p++ - captureInd_p->dcOffset.meanI)/16;
            *buff_out_p++ = (*buff_in_p++ - captureInd_p->dcOffset.meanQ)/16;
        }
    }            
}

#ifdef AOA_ALGO_INTEGRATION

#define FS 4000000
#define FS_DIV_2PI (FS / (2*1000*PI)) // 636.619772367581343
//#define ENABLE_CFO_FP

/* Estimate and apply CFO correction */
static lclStatus_t LclAoACFOcorrection(lclAoaCaptureInd_t *captureInd_p)
{
#ifdef ENABLE_CFO_FP
    int32_t Iout, Qout, tmpI, tmpQ;
    float cfoEstFP;
#else
    float cfoEst;
#endif
    lclStatus_t res;
    uint32_t Nseg = captureInd_p->ant[0].numIq; // same for all chuncks
    uint32_t switchPeriod = lclAoaGlob.ant_cnf.switchingPeriodDuration * lclAoaGlob.sampleRate;
    
    /* Estimate CFO */
#ifndef ENABLE_CFO_FP
    res = LclAoaAlgoToneFrequencyEstimate(captureInd_p->ant, Nseg, (uint32_t)captureInd_p->antPatternLength, &cfoEst);
#else
    res = LclAoaAlgoToneFrequencyEstimate_FP(captureInd_p->ant, Nseg, (uint32_t)captureInd_p->antPatternLength, &Iout, &Qout);
#endif    
    
    if (res == glclStatusSuccess)
    {
#ifdef ENABLE_CFO_FP
        /* Scale Iout and Iout to fit in an int16 */
        tmpI = Iout;
        tmpQ = Qout;
        while ((tmpI > INT16_MAX) || (tmpI < INT16_MIN) || (tmpQ > INT16_MAX) || (tmpQ < INT16_MIN))
        {
            tmpQ /= 2;
            tmpI /= 2;
        }
        
        //cfoEstFP = (float)atan2fp((int16_t)tmpQ, (int16_t)tmpI);                                                
        cfoEstFP /= Q12_SCALING_FACTOR;
        /* Convert in KHz */
        cfoEstFP = (-1) * cfoEstFP * FS_DIV_2PI;
        
        //if (cfoEstFP > cfoEst) delta = cfoEstFP - cfoEst;
        //else delta = cfoEst - cfoEstFP;
        
        /* Compute delta relative to 250 KHz and scale to Q6.9 */
        cfoEstFP = (cfoEstFP - 250) * 512 ;
        
        /* Override HW CFO estimation by this finer grain one */
        captureInd_p->cfo_est = (int16_t) cfoEstFP;
        
        /* Compensate CFO in IQ buffer used by pre-processing */
        LclAoaAlgoToneFrequencyCompensate_FP(captureInd_p->ant, Nseg, (uint32_t)captureInd_p->antPatternLength, switchPeriod, switchPeriod/2U, &Iout, &Qout);
#else  
        /* Compensate CFO in IQ buffer used by pre-processing */
        LclAoaAlgoToneFrequencyCompensate(captureInd_p->ant, Nseg, (uint32_t)captureInd_p->antPatternLength, switchPeriod, switchPeriod/2U, &cfoEst);
        
        /* Convert in KHz */
        cfoEst = (-1) * cfoEst * FS_DIV_2PI;
        
        /* Compute delta relative to 250 KHz and scale to Q6.9 */
        cfoEst = (cfoEst - 250) * 512 ;

        /* Override HW CFO estimation by this finer grain one */
        captureInd_p->cfo_est = (int16_t) cfoEst;
#endif
    }
    else
    {
        captureInd_p->cfo_est = 0; /* error */
    }
    
    return res;
}
#endif

/* Apply Antenna phase calibration to IQ data */
static void LclAoAapplyAntennaCalibration(lclAoaCaptureInd_t *captureInd_p)
{
    uint32_t patternIdx, antId, j;
    int16_t *buff_in_p;
    int32_t Isample, Qsample, temp, cos, sin;
    
    for (patternIdx = 0u ; patternIdx < captureInd_p->antPatternLength; patternIdx++)
    {    
        buff_in_p = captureInd_p->ant[patternIdx].iqBuff_p;
        antId = (uint32_t)captureInd_p->ant[patternIdx].antId;
        
        if (lclAoaGlob.antCalibration[antId].phase != 0)
        {
            cos = (int32_t)lclAoaGlob.antCalibration[antId].cos; // Q15
            sin = (int32_t)lclAoaGlob.antCalibration[antId].sin; // Q15
            
            /* Apply antenna calibration to input capture: */
            /*  (I + jQ) * exp(-j.phase_calib[antId]) */
            for (j=0; j < captureInd_p->ant[patternIdx].numIq; j++)
            {
                Isample = (int32_t)(*buff_in_p);
                Qsample = (int32_t)(*(buff_in_p + 1));
                
                temp = (Isample * cos) + (Qsample * sin);                
                *buff_in_p++ = (int16_t)(temp / Q15_SCALING_FACTOR);
                
                temp = (Qsample * cos) - (Isample * sin);
                *buff_in_p++ = (int16_t)(temp / Q15_SCALING_FACTOR);
            }
        }
        /* else no correction to be applied */
    }  
}

#ifdef AOA_ALGO_INTEGRATION
lclStatus_t LclAoAwrapperToAlgorithm(lclAoaCaptureInd_t *captureInd_p)
{
    int i, j, ant;
    lclStatus_t statusMp = glclStatusSuccess;
    uint32_t numOfpatternRepetition = (uint32_t)captureInd_p->antPatternLength / captureInd_p->numAnt;
    uint32_t Nseg = (uint32_t)captureInd_p->ant[0].numIq; // same for all chuncks
    float angEstOut[LCL_AOA_MAX_REPETITION_PATTERN];
    float aqi[LCL_AOA_MAX_REPETITION_PATTERN];
    
    if (lclAoaGlob.state != LCL_AOA_STATE_CONFIGURED)
    {  
        return glclStatusInvalidState;
    }
    
    /* Convert data to be compatible with MP algo interface ... */
    for (i = 0; i < numOfpatternRepetition; i++)
    {
        for (ant=0; ant<captureInd_p->numAnt; ant++)
        {
            for (j=0; j<Nseg; j++)
            {
               Iin[Nseg*ant + j] = ((float)captureInd_p->ant[ant+i*captureInd_p->numAnt].iqBuff_p[2*j]) / Q11_SCALING_FACTOR;
               Qin[Nseg*ant + j] = ((float)captureInd_p->ant[ant+i*captureInd_p->numAnt].iqBuff_p[2*j+1]) / Q11_SCALING_FACTOR;
            }
        }
        
        /* Call algo for each block of Nseg samples */
        statusMp = LclAoaAlgoMatrixPencilSeq(Iin, Qin, Nseg, AVERAGING_WINDOW, (uint32_t)lclAoaGlob.nMpSrc, (uint32_t)(lclAoaGlob.ant_cnf.dAnt / 2), (uint32_t)captureInd_p->numAnt, &angEstOut[i], &aqi[i], FALSE  /* fp_en */, QR_NUM_ITERATION, QR_EPSILON);
        
        if (statusMp != glclStatusSuccess)
        {
            captureInd_p->algoOut[i].angleAoA = LCL_AOA_INVALID_ANGLE;
            captureInd_p->algoOut[i].aqi = LCL_AOA_INVALID_AQI;  
            break;
        }
        
        /* Populate reporting structure with algorithm output (angle and AQI) */
        captureInd_p->algoOut[i].angleAoA = (int16_t) (angEstOut[i] * 180.0 * Q6_SCALING_FACTOR); // in degrees Q9.6 format
        captureInd_p->algoOut[i].aqi = (int16_t) (aqi[i] * Q15_SCALING_FACTOR); // AQI in Q15 format
    }
    
    if (captureInd_p->reportType == gAoAReportAngle)
    {
        captureInd_p->antPatternLength = 0;
    }
    
    return statusMp;
}
#endif

static void LclAoACaptureDmaCallback(void *userData, bool transferDone, uint16_t count)
{    
    if (transferDone)
    {
        /* Assert disabled until CONNRF-740 is fixed */
        //LCL_ASSERT(count == lclAoaGlob.numIqBuffTemp);
    
#if defined(LCL_COVERAGE)
        /* Hook */
        LclAoATweakIqSample(lclAoaGlob.captureInd_p);
#endif
        /* Check IQ saturation */
        LclAoAIqSampleCheck(lclAoaGlob.captureInd_p);
    }
    else
    {
        /* DMA error occurred */
        lclAoaGlob.captureInd_p->measStatus = gAoAMeasFailDmaError;
    }
    
    lclAoaGlob.state = LCL_AOA_STATE_CONFIGURED;

    lclAoaGlob.captureCallback(lclAoaGlob.captureInd_p);
}

static void LclAoAInitDma(void)
{
    dmaStatus_t status;
    
    dsb_config_t dsb_config;
    
    dsb_config.result_buffer = (void*)lclAoaGlob.iqBuffTemp_p; /* Output buffer pointer to store the data */
    dsb_config.buffer_sz = (uint32_t)lclAoaGlob.numIqBuffTemp; /* Output buffer size in 32-bit words */
    dsb_config.watermark = 8; /* FIFO Watermark value to perform DMA transfers */
    dsb_config.user_callback = LclAoACaptureDmaCallback;
    dsb_config.userData = NULL; /* User data available from callback */
        
    status = dma_init(&dsb_config);
        
    LCL_ASSERT(DMA_SUCCESS == status);
    NOT_USED(status);
}

/*! IQ_MUX_SEL - DMA DTEST IQ Mux Select
 *  0b0000..Decimation filter I/Q output
 *  0b0001..IQMC I/Q output
 *  0b0010..DCOC I/Q output
 *  0b0011..Channel filter I/Q output
 *  0b0100..SRC I/Q output
 *  0b0101..DC Residual I/Q output
 *  0b0110..Sample Buffer I/Q output
 *  0b0111..CFO Mixer I/Q output
 *  0b1000..Demod Channel filter I/Q output
 *  0b1001..Frac Corr I/Q output
 */
LCL_STATIC void LclAoAProgramDma(lclAoaCaptureTriggerType_t trigger)
{
    dmaStatus_t status;
    dma_capture_config_t dma_conf;

    switch(trigger)
    {
    case LCL_AOA_CAPTURE_TRIGGER_PATTERN_MATCH:
        dma_conf.start_trig = START_DMA_ON_LCL_PATT_MATCH;
        break;
    case LCL_AOA_CAPTURE_TRIGGER_CTE_PRESENT:
        dma_conf.start_trig = START_DMA_ON_GENLL_CTE_PRESENT;
        break;
    case LCL_AOA_CAPTURE_TRIGGER_AA_MATCH:
    default:
        dma_conf.start_trig = START_DMA_ON_FSK_AA_MATCH;
        break;
    }
    
    dma_conf.dma_page = DMA_PAGE_RXDIGIQ;
    dma_conf.start_delay = lclAoaGlob.dmaTriggerDelay;
    
#ifdef LCL_AOA_USE_FRAC_CORR_IQ_CAPTURE_POINT
    /* Kept for reference only */
    dma_conf.out_sel = DMA_OUT_FRAC_CORR_SEL;
    dma_conf.start_delay += LCL_DMA_DELAY_INCREMENT_US;
#else    
    /* Configure IQ capture point to be after SRC buffer. Rationale: capture must be performed before CFO mixer 
     *  in order to avoid to get the DC component shifted in frequency due to potential CFO correction */
    dma_conf.out_sel = DMA_OUT_SRC_SEL;
#endif
    
    dma_conf.decimation = DMA_NO_DECIMATE;
    dma_conf.capture_edge = DMA_START_RISING;
    
    status = dma_start_capture(&dma_conf);

    LCL_ASSERT(DMA_SUCCESS == status);
    NOT_USED(status);
}

/*!
 * @brief AGC lock/unlock state machine
 *
 * @param event input event to the state machine
 */
LCL_STATIC void LclAoaAGCupdate(lclAoaAgcEvent_t event)
{
    static uint32_t nbCapture = 0;
    static uint32_t nbOk = 0;
    uint32_t idx;
    
    switch (lclAoaGlob.agcState)
    {
    case LCL_AOA_AGC_STATE_UNLOCKED:
      nbCapture = 0;
      nbOk = 0;
      lclAoaGlob.agcState = LCL_AOA_AGC_STATE_LOCKING;
      // intentional drop trough
      
    case LCL_AOA_AGC_STATE_LOCKING:
      if (event == LCL_AOA_AGC_EVENT_CAPTURE_OK)
      {
          nbOk++;
      }
      else // if (event == LCL_AOA_AGC_EVENT_SATURATION_DETECTED)
      {
          if (nbOk > 0U)
          {
              nbOk--;
          }
      }
      
      /* Store AGC index in table */
      LCL_AOA_AGC_RECORD_IDX(idx, nbCapture);
      nbCapture++;
      
      if (nbOk == LCL_AOA_NUM_CAPTURE_BEFORE_AGC_LOCK)
      {
          /* Compute AGC index average from table */
          LCL_AOA_AGC_COMPUTE_MEAN(idx, nbCapture);
          
          /* Lock AGC and move to next state */
          lclAoaGlob.forcedAgcIndex = (uint8_t)idx;
          (void)LCL_lockAgc(lclAoaGlob.forcedAgcIndex);
          lclAoaGlob.agcState = LCL_AOA_AGC_STATE_LOCKED;
          nbOk = 0;
          nbCapture = 0;
      }
      if (nbCapture == LCL_AOA_MAX_CAPTURE_DURING_AGC_LOCK)
      {
          /* Compute AGC index average from table */
          LCL_AOA_AGC_COMPUTE_MEAN(idx, nbCapture);
          
          /* Step down and lock AGC and reset state machine */
          if (idx > 0U)
          {
              idx--;
          }
          lclAoaGlob.forcedAgcIndex = (uint8_t)idx;
          (void)LCL_lockAgc(lclAoaGlob.forcedAgcIndex);
          lclAoaGlob.agcState = LCL_AOA_AGC_STATE_UNLOCKED;
      }
      break;
      
    case LCL_AOA_AGC_STATE_LOCKED:
      if (event == LCL_AOA_AGC_EVENT_SATURATION_DETECTED)
      {
          nbCapture++;
      }
      else // if (event == LCL_AOA_AGC_EVENT_CAPTURE_OK)
      {
          nbOk++;
          if (nbCapture > 0U)
          {
              nbCapture--;
          }
      }

      if (nbCapture == LCL_AOA_NUM_CAPTURE_BEFORE_AGC_UNLOCK)
      {
          /* Step down AGC by 1 step and reset state */
          if (lclAoaGlob.forcedAgcIndex > 0U) 
          {
              lclAoaGlob.forcedAgcIndex--;
          }
          (void)LCL_lockAgc(lclAoaGlob.forcedAgcIndex);
          lclAoaGlob.agcState = LCL_AOA_AGC_STATE_UNLOCKED;
      }
      break;
//LCOV_EXCL_START
      default:
        /* MISRA rule 16.4 */
        break;
//LCOV_EXCL_STOP
    }
}

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

/* configure AoA */
lclStatus_t LclAoAConfigureIqCapture(lclAoaConfig_t *config_p)
{
    lclStatus_t status = glclStatusSuccess;
    uint32_t i;
    uint16_t numTemp;

    if (lclAoaGlob.state == LCL_AOA_STATE_IDLE)
    {
        LCL_ASSERT(config_p->sampleRate == 4U);

        /* Copy config */
        lclAoaGlob.sampleRate = config_p->sampleRate;
        lclAoaGlob.nMpSrc = config_p->nMpSrc;
        lclAoaGlob.dmaTriggerDelay = config_p->dmaTriggerDelay;
        lclAoaGlob.captureTrigger = config_p->captureTrigger;
        lclAoaGlob.captureCallback = config_p->captureCallback;
        lclAoaGlob.antSwitchCallback = config_p->antSwitchCallback;
        lclAoaGlob.agcMode = config_p->agcMode;
        lclAoaGlob.disableCFOest = config_p->disableCFOest;
        
        lclAoaGlob.byteStreamPatternSize = config_p->byteStreamPatternSize;
        for (i=0; i<lclAoaGlob.byteStreamPatternSize; i++)
        {
            lclAoaGlob.byteStreamPattern[i] = config_p->byteStreamPattern[i];
        }
        lclAoaGlob.ant_cnf = config_p->ant_cnf; // structure copy

        /* Allocate IQ buffers */
        numTemp = ((lclAoaGlob.ant_cnf.numOfpatternRepetition * lclAoaGlob.ant_cnf.numAnt * (lclAoaGlob.ant_cnf.switchingPeriodDuration + lclAoaGlob.ant_cnf.samplingPeriodDuration))
                    + lclAoaGlob.ant_cnf.guardPeriodDuration) * lclAoaGlob.sampleRate;
        lclAoaGlob.numIqBuffTemp = numTemp;

        LCL_ASSERT(lclAoaGlob.iqBuffTemp_p == NULL);
        LCL_ASSERT(lclAoaGlob.captureInd_p == NULL);
        LCL_ASSERT(lclAoaGlob.iqBuff_p == NULL);

        /* Temp IQ buffer to store DMA output */
        lclAoaGlob.iqBuffTemp_p =  (uint32_t *)LCLMemAlloc(((uint32_t)numTemp) * LCL_IQ_SAMPLE_LENGTH);
        LCL_ASSERT(lclAoaGlob.iqBuffTemp_p != NULL);
        
        /* IQ buffer to execute pre-processing */
        numTemp -= (lclAoaGlob.ant_cnf.guardPeriodDuration * lclAoaGlob.sampleRate);
        lclAoaGlob.iqBuff_p = (int16_t*)LCLMemAlloc(((uint32_t)numTemp) * LCL_IQ_SAMPLE_LENGTH);
        LCL_ASSERT(lclAoaGlob.iqBuff_p != NULL);

        /* Allocate report structure */
        lclAoaGlob.captureInd_p =  (lclAoaCaptureInd_t *)LCLMemAlloc(LCL_CAPTURE_IND_MSG_SIZE(((uint32_t)lclAoaGlob.ant_cnf.numOfpatternRepetition) * ((uint32_t)lclAoaGlob.ant_cnf.numAnt)));
        LCL_ASSERT(lclAoaGlob.captureInd_p != NULL);
        
        /* Apply default calibration (no calibration) */
        if (!lclAoaGlob.calibrationDone)
        {
            for (i=0; i<LCL_AOA_NUM_MAX_ANTENNAS; i++)
            {
                lclAoaGlob.antCalibration[i].cos = 1;
                lclAoaGlob.antCalibration[i].sin = 0;
                lclAoaGlob.antCalibration[i].phase = 0;
            }
        }
        
        /* Initialize states */
        lclAoaGlob.state = LCL_AOA_STATE_CONFIGURED;
        
        switch (lclAoaGlob.agcMode)
        {
        case LCL_AOA_AGC_MODE_MANUAL:
            lclAoaGlob.forcedAgcIndex = 0;
            lclAoaGlob.agcState = LCL_AOA_AGC_STATE_UNLOCKED;
            break;
            
        case LCL_AOA_AGC_MODE_FORCED:
            lclAoaGlob.forcedAgcIndex = config_p->forcedAgcIndex;
            (void)LCL_lockAgc(lclAoaGlob.forcedAgcIndex);
            lclAoaGlob.agcState = LCL_AOA_AGC_STATE_LOCKED;
            break;
            
        case LCL_AOA_AGC_MODE_AUTO:
        default:
            lclAoaGlob.agcState = LCL_AOA_AGC_STATE_AUTO;
            break;
        }
    }
    else
    {
        status = glclStatusInvalidState;
    }

    /* Initialize antenna swicthing and DMA (modules will remain configured for each capture) */
    if (status == glclStatusSuccess)
    {
        LclAoAInitAntennaSwitching();
        LclAoAInitDma();
    }

    return status;
}

/* disable AoA */
lclStatus_t LclAoADisableIqCapture(void)
{
    /* Stop any potential active capture */
    LclAoAStopCapture();

    /* Deinit DMA and LCL modules */
    LclAoADeinitAntennaSwitching();

    dma_release();

    /* Free-up memory */
    if (NULL != lclAoaGlob.iqBuffTemp_p)
    {
        (void)LCLMemFree(lclAoaGlob.iqBuffTemp_p);
        lclAoaGlob.iqBuffTemp_p = NULL;
    }
    if (NULL != lclAoaGlob.iqBuff_p)
    {
        (void)LCLMemFree(lclAoaGlob.iqBuff_p);
        lclAoaGlob.iqBuff_p = NULL;
    }
    if (NULL != lclAoaGlob.captureInd_p)
    {
        (void)LCLMemFree(lclAoaGlob.captureInd_p);
        lclAoaGlob.captureInd_p = NULL;
    }
    
    lclAoaGlob.state = LCL_AOA_STATE_IDLE;
    
    /* Unlock AGC and reset state */
    if (lclAoaGlob.agcMode != LCL_AOA_AGC_MODE_AUTO)
    {
        LCL_unlockAgc();
    }

    return glclStatusSuccess;
}

/* start AoA capture */
void LclAoAStartCapture(lclAoaStartCapture_t *startCapture_p)
{
    uint16_t patterdIdx, j;
    
    LCL_ASSERT(lclAoaGlob.state == LCL_AOA_STATE_CONFIGURED);
    
    /* Prepare and initialize reporting structure */
    lclAoaGlob.captureInd_p->measId = startCapture_p->measId;
    lclAoaGlob.captureInd_p->reportType = startCapture_p->reportType;
    lclAoaGlob.captureInd_p->polarConfig = startCapture_p->polarConfig;
    lclAoaGlob.captureInd_p->numAnt = lclAoaGlob.ant_cnf.numAnt;
    lclAoaGlob.captureInd_p->measStatus = gAoAMeasSuccess;
    lclAoaGlob.captureInd_p->iqBuffLength = lclAoaGlob.numIqBuffTemp;
    lclAoaGlob.captureInd_p->rawIqBuff_p = (int16_t *)lclAoaGlob.iqBuffTemp_p;
    
    for (j=0; j<LCL_AOA_MAX_REPETITION_PATTERN; j++)
    {
        lclAoaGlob.captureInd_p->algoOut[j].angleAoA = (int16_t)LCL_AOA_INVALID_ANGLE;
        lclAoaGlob.captureInd_p->algoOut[j].aqi = (int16_t)LCL_AOA_INVALID_AQI;
    }
    
    if (lclAoaGlob.captureInd_p->reportType == gAoAReportIqRaw)
    {
        /* All captured IQ samples will be provided to the application in ant[0].iqBuff_p */
        lclAoaGlob.captureInd_p->antPatternLength = 1;
        lclAoaGlob.captureInd_p->ant[0].antId = lclAoaGlob.ant_cnf.mainAntenna;
        lclAoaGlob.captureInd_p->ant[0].numIq = lclAoaGlob.numIqBuffTemp;
        lclAoaGlob.captureInd_p->ant[0].iqBuff_p = lclAoaGlob.captureInd_p->rawIqBuff_p;
        lclAoaGlob.captureInd_p->ant[0].antMeasStatus = gAoAMeasSuccess;
    }
    else
    {
        /* Extract IQ samples from capture buffer and keep only samples corresponding to gSamplingPeriodDuration for each antenna switch */
        /* Samples corresponding to gSwitchingPeriodDuration are discarded (half before and after antenna switch */
        for (patterdIdx = 0u ; patterdIdx < lclAoaGlob.ant_cnf.numOfpatternRepetition * lclAoaGlob.ant_cnf.numAnt; patterdIdx++)
        {
            j = (lclAoaGlob.ant_cnf.guardPeriodDuration + 
                + (patterdIdx * (lclAoaGlob.ant_cnf.switchingPeriodDuration + lclAoaGlob.ant_cnf.samplingPeriodDuration)) 
                + (lclAoaGlob.ant_cnf.switchingPeriodDuration / 2u)) * lclAoaGlob.sampleRate;
            
            lclAoaGlob.captureInd_p->ant[patterdIdx].antId = lclAoaGlob.ant_cnf.antPattern[patterdIdx % lclAoaGlob.ant_cnf.numAnt];
            lclAoaGlob.captureInd_p->ant[patterdIdx].numIq = lclAoaGlob.ant_cnf.samplingPeriodDuration * lclAoaGlob.sampleRate;
            lclAoaGlob.captureInd_p->ant[patterdIdx].iqBuff_p = (int16_t *) &lclAoaGlob.iqBuffTemp_p[j];
        }
        lclAoaGlob.captureInd_p->antPatternLength = lclAoaGlob.ant_cnf.numOfpatternRepetition * (uint16_t)lclAoaGlob.ant_cnf.numAnt;
    }
    
    lclAoaGlob.state = LCL_AOA_STATE_CAPTURING;

    /* GENFSK should have been configured in Rx at this point */
    if (lclAoaGlob.disableCFOest)
    {
        LCL_enableCFOestimateOverride(1U);
    }
    
    /* Program Antenna switching */ 
    LclAoAStartAntennaSwitching();
    
    /* Program DMA for IQ capture */
    LclAoAProgramDma(lclAoaGlob.captureTrigger);
}

/* stop AoA capture */
void LclAoAStopCapture(void)
{
    /* Stop LCL module */
    LclAoAStopAntennaSwitching();
    
    /* And abort DMA if needed */
    if (lclAoaGlob.state == LCL_AOA_STATE_CAPTURING)
    {
        /* Abort and reset DMA */
        dma_abort();
        LclAoAInitDma();
    }
    
    if (lclAoaGlob.disableCFOest)
    {
        LCL_disableCFOestimateOverride();
    }
    lclAoaGlob.state = LCL_AOA_STATE_CONFIGURED;
}


/* request pre-processing of IQ samples taken during last capture */
lclStatus_t LclAoAIqSamplePreProcessing(uint32_t flags, lclAoaCaptureInd_t *captureInd_p)
{
    lclStatus_t status = glclStatusSuccess;
    
    /* Update AGC state machine if needed */
    LclAoAUpdateAGCstateMachine(captureInd_p);
    
    if (lclAoaGlob.captureInd_p->reportType != gAoAReportIqRaw)
    {
        DEBUG_DWT_CYCLE_CNT_START();
        
        if ((flags & LCL_AOA_PREPROC_DC_OFFSET_REMOVAL) != 0UL)
        {
            /* Estimate DC offset using guard period */
            LclAoAEstimateDcOffset(captureInd_p);
            
            /* Remap IQ data and remove DC offset */
            LclAoARemapIQdata(captureInd_p);
        }
        
        DEBUG_DWT_CYCLE_CNT_FETCH(dwt_preproc);
        
#ifdef AOA_ALGO_INTEGRATION
        if ((flags & LCL_AOA_PREPROC_CFO_REMOVAL) != 0UL)
        {    
            /* CFO estimation and correction */
            status = LclAoACFOcorrection(captureInd_p);
        }
        
        DEBUG_DWT_CYCLE_CNT_FETCH(dwt_preproc);
#endif
        if ((status == glclStatusSuccess) &&
           ((flags & LCL_AOA_PREPROC_ANTENNA_CAL_COMPENSATION) != 0UL))
        {
            /* Apply antenna calibration */
            LclAoAapplyAntennaCalibration(captureInd_p);
        }
        
        DEBUG_DWT_CYCLE_CNT_FETCH(dwt_preproc);
    }
       
    return status;

}


/* request post-processing of IQ samples and execute AoA algorithm */
lclStatus_t LclAoAIqSamplePostProcessing(lclAoaCaptureInd_t *captureInd_p)
{
    lclStatus_t status = glclStatusSuccess;
    
#ifdef AOA_ALGO_INTEGRATION
    /* Call MP algorithm */
    if ((captureInd_p->reportType == gAoAReportAngleAndIqFiltered) || (captureInd_p->reportType == gAoAReportAngle) ||
        (captureInd_p->reportType == gAoAReportAngleAndIqRaw))
    {  
        DEBUG_DWT_CYCLE_CNT_START();
        
        status = LclAoAwrapperToAlgorithm(captureInd_p);
        
        DEBUG_DWT_CYCLE_CNT_FETCH(dwt_MP);
    }
#endif
    
    return status;
}


/* get computed phase calibration vector */
void LclAoAgetPhaseCalibration(Q15_t cal_phase[LCL_AOA_NUM_MAX_ANTENNAS])
{
    uint32_t i;
    
    for (i=0; i<lclAoaGlob.ant_cnf.numAnt; i++)
    {
        cal_phase[i] = lclAoaGlob.antCalibration[i].phase;
    }
}


/* set computed phase calibration vector */
void LclAoAsetPhaseCalibration(Q15_t cal_phase[LCL_AOA_NUM_MAX_ANTENNAS])
{
    uint32_t i;
    
    for (i=0; i<lclAoaGlob.ant_cnf.numAnt; i++)
    {
        LCL_ASSERT(cal_phase[i] >= 0);
        
        /* Compute Cosinus and Sinus */
        if (cal_phase[i] != 0)
        {
            /* ARM math lib is used */
            lclAoaGlob.antCalibration[i].cos = arm_cos_q15(cal_phase[i]);
            lclAoaGlob.antCalibration[i].sin = arm_sin_q15(cal_phase[i]);
        }
        else
        {
            lclAoaGlob.antCalibration[i].cos = 1;
            lclAoaGlob.antCalibration[i].sin = 0;
        }
        
        /* Store phase */
        lclAoaGlob.antCalibration[i].phase = cal_phase[i];
    }
    
    lclAoaGlob.calibrationDone = TRUE;
}


/* S(3,12) fixed point format */
#define Q3p12_PI          (12867)
#define Q3p12_2PI         (25735)
#define Q3p12_MINUS_PI    (-12867)

/* request computation phase calibration vector */
lclStatus_t LclAoAcomputePhaseCalibration(lclAoaCaptureInd_t *captureInd_p, Q15_t phaseArray[LCL_AOA_NUM_MAX_ANTENNAS])
{
    lclStatus_t status = glclStatusSuccess;
    
    uint32_t patternIdx, antId, k;
    int16_t *buff_in_p;
    uint16_t num_iq_per_ant = captureInd_p->ant[0].numIq * lclAoaGlob.ant_cnf.numOfpatternRepetition;
    int32_t cal_phase;
    int32_t phase_vector[LCL_AOA_NUM_MAX_ANTENNAS];
    int32_t I_avg[LCL_AOA_NUM_MAX_ANTENNAS];
    int32_t Q_avg[LCL_AOA_NUM_MAX_ANTENNAS];
    
    /* Initialize accumulator */
    for (antId=0; antId<LCL_AOA_NUM_MAX_ANTENNAS; antId++)
    {
        I_avg[antId] = 0;
        Q_avg[antId] = 0;
    }
    
    /* Accumulate I and Q samples for all pattern repetition and group by antenna */
    for (patternIdx = 0u ; patternIdx < captureInd_p->antPatternLength; patternIdx++)
    {    
        buff_in_p = captureInd_p->ant[patternIdx].iqBuff_p;
        antId = (uint32_t)captureInd_p->ant[patternIdx].antId;
        
        for (k = 0u ; k < captureInd_p->ant[patternIdx].numIq; k++)
        {
            I_avg[antId] += *buff_in_p++;
            Q_avg[antId] += *buff_in_p++;
        }
    }
    
    LCL_ASSERT(num_iq_per_ant != 0);
    
    /* Now compute phase per antenna */
    for (antId=0; antId<captureInd_p->numAnt; antId++)
    {   
        /* Compute average */
        I_avg[antId] /= (int32_t)num_iq_per_ant;
        Q_avg[antId] /= (int32_t)num_iq_per_ant;
        
        /* compute cal_phase in Q3.12 ranging from -pi to +pi */
        cal_phase = (int32_t) atan2fp((int16_t)Q_avg[antId], (int16_t)I_avg[antId]);
        
        if (cal_phase < 0) 
        {
            cal_phase += Q3p12_2PI;
        }
        
        phase_vector[antId] = cal_phase; // Q3.12 ranging from 0 to 2*pi */
        
        /* Substract phase from main antenna */
        cal_phase -= phase_vector[0];
        
        if (cal_phase < 0) 
        {
            cal_phase += Q3p12_2PI;
        }
        if (cal_phase >= Q3p12_2PI)
        {
            cal_phase -= Q3p12_2PI;
        }
        
        /* store phase calibration is in radians [0; 2pi) scaled normalized to [0; +1) in Q15 format */ 
        cal_phase = (cal_phase * Q15_SCALING_FACTOR) / Q3p12_2PI;
        
        phaseArray[antId] = (Q15_t) cal_phase;
    }
    
    return status;
}
